<?php
/**
 * Handles URL list functionality for health issues.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Rank_URL_List {

    /**
     * Initialize hooks.
     */
    public static function init() {
        add_action('wp_ajax_rank_get_issue_urls', array(__CLASS__, 'ajax_get_issue_urls'));
        add_action('wp_ajax_rank_get_fixed_issue_urls', array(__CLASS__, 'ajax_get_fixed_issue_urls'));
    }

    /**
     * Extract issue details (problem identifier and page URL) from an issue array.
     *
     * @param array  $issue     The issue data.
     * @param string $issue_type The issue type key.
     * @return array Associative array with 'problem' and 'url' keys.
     */
    private static function extract_issue_details($issue, $issue_type = '') {
        $page_url = '';
        $problem = '';

        // Extract page URL based on issue structure
        if (isset($issue['page']['path_full'])) {
            $page_url = $issue['page']['path_full'];
        } elseif (isset($issue['page']['url'])) {
            $page_url = $issue['page']['url'];
        } elseif (isset($issue['url'])) {
            $page_url = $issue['url'];
        }

        // Extract the problem identifier
        $issue_identifier = isset($issue['issue_identifier']) ? $issue['issue_identifier'] : '';
        $parameters_url = isset($issue['parameters']['url']) ? $issue['parameters']['url'] : '';

        // Determine issue type config to know if identifier_is_url
        $issue_config = function_exists('rank_get_issue_type_config') ? rank_get_issue_type_config($issue_type) : null;
        $identifier_is_url = ($issue_config && isset($issue_config['identifier_is_url'])) ? $issue_config['identifier_is_url'] : false;
        
        // Check if this is a broken link issue type
        $is_broken_link = in_array($issue_type, array('broken_links_internal', 'broken_links_external', 'broken_internal_links', 'broken_external_links'));

        if ($identifier_is_url) {
            if ($is_broken_link) {
                // For broken links: try multiple sources for the actual broken URL
                // Priority: parameters.url > issue_identifier (if URL) > decoded rank_id > issue_identifier (anchor text)
                if (!empty($parameters_url)) {
                    $problem = $parameters_url;
                } elseif (!empty($issue_identifier) && filter_var($issue_identifier, FILTER_VALIDATE_URL)) {
                    $problem = $issue_identifier;
                } else {
                    // issue_identifier is likely anchor text (e.g. "power-words")
                    // Try to decode rank_id to extract the actual broken URL
                    $decoded_url = self::decode_rank_id_for_url($issue);
                    if (!empty($decoded_url) && $decoded_url !== $issue_identifier) {
                        // Found actual URL from rank_id - show both
                        $problem = $decoded_url;
                    } elseif (!empty($issue_identifier)) {
                        // No URL found - show anchor text with label
                        $problem = 'Link text: ' . $issue_identifier;
                    }
                }
            } elseif (!empty($issue_identifier)) {
                // For alt tags and other URL-based identifiers: use issue_identifier directly
                $problem = $issue_identifier;
            }
        }
        // For meta/title issues (identifier_is_url = false): problem stays empty

        return array(
            'problem' => $problem,
            'url' => $page_url,
            'rank_id' => isset($issue['rank_id']) ? $issue['rank_id'] : '',
            'title' => isset($issue['page']['title']) ? $issue['page']['title'] : basename($page_url)
        );
    }

    /**
     * Try to decode a rank_id (base64 JSON) to extract the actual broken URL.
     * The rank_id may contain a payload with issue_identifier that is a URL.
     *
     * @param array $issue The issue data containing 'rank_id'.
     * @return string The decoded URL, or empty string if not found.
     */
    private static function decode_rank_id_for_url($issue) {
        $rank_id = isset($issue['rank_id']) ? $issue['rank_id'] : '';
        if (empty($rank_id)) {
            return '';
        }

        // Fix base64 padding if needed
        $rank_id_padded = $rank_id;
        $mod = strlen($rank_id_padded) % 4;
        if ($mod) {
            $rank_id_padded .= str_repeat('=', 4 - $mod);
        }

        $decoded = base64_decode($rank_id_padded);
        if ($decoded === false || empty($decoded)) {
            return '';
        }

        $json_data = json_decode($decoded, true);
        if (!is_array($json_data)) {
            return '';
        }

        // Try direct issue_identifier
        if (isset($json_data['issue_identifier']) && filter_var($json_data['issue_identifier'], FILTER_VALIDATE_URL)) {
            return $json_data['issue_identifier'];
        }

        // Try nested payload
        if (isset($json_data['payload'])) {
            $payload = json_decode($json_data['payload'], true);
            if (is_array($payload)) {
                if (isset($payload['issue_identifier']) && filter_var($payload['issue_identifier'], FILTER_VALIDATE_URL)) {
                    return $payload['issue_identifier'];
                }
                // Try parameters.url in payload
                if (isset($payload['parameters']['url']) && filter_var($payload['parameters']['url'], FILTER_VALIDATE_URL)) {
                    return $payload['parameters']['url'];
                }
            }
        }

        // Try parameters.url in decoded data
        if (isset($json_data['parameters']['url']) && filter_var($json_data['parameters']['url'], FILTER_VALIDATE_URL)) {
            return $json_data['parameters']['url'];
        }

        return '';
    }

    /**
     * AJAX handler for getting URLs from a batch (unfixed issues).
     */
    public static function ajax_get_issue_urls() {
        // Verify nonce and permissions
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rank_ajax_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }

        // Get and validate batch ID
        $batch_id = isset($_POST['batch_id']) ? sanitize_key($_POST['batch_id']) : '';
        if (empty($batch_id) || strpos($batch_id, 'rank_batch_') !== 0) {
            wp_send_json_error(array('message' => 'Invalid batch ID'));
            return;
        }

        // Get page and per_page parameters
        $page = isset($_POST['page']) ? max(1, intval($_POST['page'])) : 1;
        $per_page = isset($_POST['per_page']) ? min(200, max(1, intval($_POST['per_page']))) : 200;

        // Get batch data
        $batch_data = get_option($batch_id);
        if (!$batch_data || !isset($batch_data['issues']) || !is_array($batch_data['issues'])) {
            wp_send_json_error(array('message' => 'Batch data not found or invalid'));
            return;
        }

        $issue_type = isset($batch_data['issue_type']) ? $batch_data['issue_type'] : '';

        // Extract issue details from issues
        $items = array();
        foreach ($batch_data['issues'] as $issue) {
            $details = self::extract_issue_details($issue, $issue_type);
            if (!empty($details['url'])) {
                $items[] = $details;
            }
        }

        // Calculate pagination
        $total = count($items);
        $total_pages = ceil($total / $per_page);
        $offset = ($page - 1) * $per_page;
        $page_items = array_slice($items, $offset, $per_page);

        // Send response
        wp_send_json_success(array(
            'urls' => $page_items,
            'total' => $total,
            'total_pages' => $total_pages,
            'current_page' => $page,
            'has_problems' => !empty($page_items) && !empty($page_items[0]['problem'])
        ));
    }

    /**
     * AJAX handler for getting already-fixed issue URLs from a batch.
     */
    public static function ajax_get_fixed_issue_urls() {
        // Verify nonce and permissions
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rank_ajax_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }

        // Get and validate batch ID
        $batch_id = isset($_POST['batch_id']) ? sanitize_key($_POST['batch_id']) : '';
        if (empty($batch_id) || strpos($batch_id, 'rank_batch_') !== 0) {
            wp_send_json_error(array('message' => 'Invalid batch ID'));
            return;
        }

        // Get page and per_page parameters
        $page = isset($_POST['page']) ? max(1, intval($_POST['page'])) : 1;
        $per_page = isset($_POST['per_page']) ? min(200, max(1, intval($_POST['per_page']))) : 200;

        // Get batch data
        $batch_data = get_option($batch_id);
        if (!$batch_data) {
            wp_send_json_error(array('message' => 'Batch data not found'));
            return;
        }

        // Get fixed issues (stored separately during filtering)
        $fixed_issues = isset($batch_data['fixed_issues']) && is_array($batch_data['fixed_issues']) 
            ? $batch_data['fixed_issues'] 
            : array();

        if (empty($fixed_issues)) {
            wp_send_json_error(array('message' => 'No fixed issues found in batch'));
            return;
        }

        $issue_type = isset($batch_data['issue_type']) ? $batch_data['issue_type'] : '';

        // Extract issue details from fixed issues
        $items = array();
        foreach ($fixed_issues as $issue) {
            $details = self::extract_issue_details($issue, $issue_type);
            if (!empty($details['url'])) {
                $items[] = $details;
            }
        }

        // Calculate pagination
        $total = count($items);
        $total_pages = ceil($total / $per_page);
        $offset = ($page - 1) * $per_page;
        $page_items = array_slice($items, $offset, $per_page);

        // Send response
        wp_send_json_success(array(
            'urls' => $page_items,
            'total' => $total,
            'total_pages' => $total_pages,
            'current_page' => $page,
            'has_problems' => !empty($page_items) && !empty($page_items[0]['problem'])
        ));
    }
}
