/**
 * URL List Modal Module
 * Handles displaying a modal with paginated URLs from health issues
 * Supports both unfixed (ready to fix) and already-fixed issue lists
 * Shows a "Problem" column for issue types that have item-specific identifiers
 */
(function($, rankLocalizedData) {
    'use strict';
    
    /**
     * Issue types that have a separate "problem" identifier (not just a page URL)
     */
    var ISSUE_TYPES_WITH_PROBLEMS = ['missing_alt_tags', 'broken_links_external', 'broken_links_internal', 'broken_external_links', 'broken_internal_links', 'image_alt'];
    
    /**
     * Initialize the module
     */
    function init() {
        // Add CSS for the modal
        addModalStyles();
    }
    
    /**
     * Check if an issue type has item-specific problems (vs page-level issues)
     * @param {string} issueType The issue type
     * @returns {boolean}
     */
    function issueTypeHasProblems(issueType) {
        return ISSUE_TYPES_WITH_PROBLEMS.indexOf(issueType) !== -1;
    }
    
    /**
     * Add the "See list" links for both fixed and unfixed issues
     * Called after issue preparation and filtering is complete
     * @param {string} batchId The batch ID
     * @param {string} issueType The issue type
     * @param {number} remainingCount Number of unfixed issues
     * @param {number} fixedCount Number of already-fixed issues
     * @param {string} healthIssueName The display name for the health issue (e.g., "Missing Alt Tag: ")
     */
    function addIssueCountsDisplay(batchId, issueType, remainingCount, fixedCount, healthIssueName) {
        if (!rank.elements.counts || !rank.elements.counts.length) {
            return;
        }
        
        // Remove any existing links
        $('.rank-view-urls-link, .rank-view-fixed-link').remove();
        
        var totalCount = remainingCount + fixedCount;
        
        // Build 3-line display
        var html = '';
        
        // Line 1: Total count
        html += '<div class="rank-issue-line rank-issue-line-total">';
        html += healthIssueName + '<span style="color: #dc3545 !important; font-weight: bold;">' + totalCount + '</span> problems total.';
        html += '</div>';
        
        // Line 2: Already fixed (if any)
        if (fixedCount > 0) {
            html += '<div class="rank-issue-line rank-issue-line-fixed">';
            html += '✅ <span style="color: #28a745 !important; font-weight: bold;">' + fixedCount + '</span> are already fixed: ';
            html += '<a href="#" class="rank-view-fixed-link" style="color: #0073aa; text-decoration: none;">See list</a>';
            html += '</div>';
        }
        
        // Line 3: Ready to fix (if any)
        if (remainingCount > 0) {
            html += '<div class="rank-issue-line rank-issue-line-remaining">';
            html += '🔧 <span style="color: #dc3545 !important; font-weight: bold;">' + remainingCount + '</span> are ready to be fixed: ';
            html += '<a href="#" class="rank-view-urls-link" style="color: #0073aa; text-decoration: none;">See list</a>';
            html += '</div>';
        }
        // When all are already fixed (0 remaining), the infobox below the buttons will explain why
        
        rank.elements.counts.html(html);
        
        // Attach click handlers
        rank.elements.counts.find('.rank-view-fixed-link').on('click', function(e) {
            e.preventDefault();
            showFixedIssuesModal(batchId, issueType);
        });
        
        rank.elements.counts.find('.rank-view-urls-link').on('click', function(e) {
            e.preventDefault();
            showUrlsModal(batchId, issueType);
        });
    }
    
    /**
     * Legacy method - Add the "View URLs" button after issue preparation
     * Now calls addIssueCountsDisplay internally
     * @param {string} batchId The batch ID
     * @param {string} issueType The issue type
     */
    function addViewUrlsButton(batchId, issueType) {
        // Use new display if we have the needed state data
        if (rank.state && typeof rank.state.alreadyFixedCount !== 'undefined' && rank.state.alreadyFixedCount > 0) {
            var healthIssueName = '';
            if (issueType && rankLocalizedData && rankLocalizedData.healthCheckCategories &&
                rankLocalizedData.healthCheckCategories[issueType]) {
                healthIssueName = rankLocalizedData.healthCheckCategories[issueType] + ': ';
            }
            addIssueCountsDisplay(batchId, issueType, rank.state.totalCount, rank.state.alreadyFixedCount, healthIssueName);
            return;
        }
        
        // Fallback: just add a simple "View pages" link (for cases with no fixed items)
        var $viewUrlsButton = $('<a>', {
            href: '#',
            class: 'rank-view-urls-link',
            text: 'View pages',
            style: 'margin-left: 10px; color: #0073aa; text-decoration: none; font-size: 14px;'
        }).on('click', function(e) {
            e.preventDefault();
            showUrlsModal(batchId, issueType);
        });
        
        // Add the button inline with the text in the counts element
        if (rank.elements.counts && rank.elements.counts.length) {
            // First remove any existing button
            $('.rank-view-urls-link').remove();
            
            // Get the current HTML from the counts element (preserves color spans)
            var countsHtml = rank.elements.counts.html();
            
            // Clear the counts element
            rank.elements.counts.empty();
            
            // Add the HTML and button directly next to each other
            rank.elements.counts.html(countsHtml);
            rank.elements.counts.append($viewUrlsButton);
        }
    }
    
    /**
     * Show the modal with unfixed URLs (ready to fix)
     * @param {string} batchId The batch ID
     * @param {string} issueType The issue type
     */
    function showUrlsModal(batchId, issueType) {
        showModal(batchId, issueType, 'unfixed');
    }
    
    /**
     * Show the modal with already-fixed issues
     * @param {string} batchId The batch ID
     * @param {string} issueType The issue type
     */
    function showFixedIssuesModal(batchId, issueType) {
        showModal(batchId, issueType, 'fixed');
    }
    
    /**
     * Show the modal with URLs
     * @param {string} batchId The batch ID
     * @param {string} issueType The issue type
     * @param {string} mode 'unfixed' or 'fixed'
     */
    function showModal(batchId, issueType, mode) {
        var modalId = 'rank-urls-modal';
        var $modal = $('#' + modalId);
        
        // Always recreate modal to update title and content for the current mode
        if ($modal.length > 0) {
            $modal.remove();
        }
        
        var title = mode === 'fixed' 
            ? 'Already Fixed Issues' 
            : 'Issues Ready to Fix';
        
        var helpLink = mode === 'fixed'
            ? ''
            : '<p><a href="https://help.morningscore.io/en/article/how-to-make-the-tool-ignore-health-issues-oyh8cj/" target="_blank" style="font-weight: bold;">⏭️ Skip problems from this list</a></p>';
        
        $modal = $('<div>', {
            id: modalId,
            class: 'rank-modal'
        }).appendTo('body');
        
        $modal.html(
            '<div class="rank-modal-content">' +
                '<span class="rank-modal-close">&times;</span>' +
                '<h3>' + title + '</h3>' +
                helpLink +
                '<div class="rank-urls-list"></div>' +
                '<div class="rank-pagination"></div>' +
            '</div>'
        );
        
        // Close button functionality
        $modal.find('.rank-modal-close').on('click', function() {
            $modal.hide();
        });
        
        // Close when clicking outside the modal
        $(window).on('click.rankModal', function(event) {
            if ($(event.target).is($modal)) {
                $modal.hide();
            }
        });
        
        // Show modal and load first page
        $modal.show();
        loadUrlsPage(batchId, issueType, 1, mode);
    }
    
    /**
     * Load a page of URLs
     * @param {string} batchId The batch ID
     * @param {string} issueType The issue type
     * @param {number} page The page number to load
     * @param {string} mode 'unfixed' or 'fixed'
     */
    function loadUrlsPage(batchId, issueType, page, mode) {
        var $urlsList = $('#rank-urls-modal .rank-urls-list');
        var $pagination = $('#rank-urls-modal .rank-pagination');
        var perPage = 200; // Items per page
        
        // Show loading
        $urlsList.html('<p><span class="spinner is-active"></span> Loading issues...</p>');
        
        // Choose AJAX action based on mode
        var ajaxAction = mode === 'fixed' ? 'rank_get_fixed_issue_urls' : 'rank_get_issue_urls';
        
        // Make AJAX request to get URLs for this page
        $.ajax({
            url: rankLocalizedData.ajax_url,
            type: 'POST',
            data: {
                action: ajaxAction,
                nonce: rankLocalizedData.nonce,
                batch_id: batchId,
                page: page,
                per_page: perPage
            },
            success: function(response) {
                if (response.success && response.data) {
                    // Determine if we should show the Problem column
                    var showProblems = issueTypeHasProblems(issueType) && response.data.has_problems;
                    
                    // Display URLs with page and perPage information for correct numbering
                    displayUrls($urlsList, response.data.urls, page, perPage, showProblems);
                    
                    // Update pagination
                    updatePagination($pagination, page, response.data.total_pages, function(newPage) {
                        loadUrlsPage(batchId, issueType, newPage, mode);
                    });
                } else {
                    var errorMsg = (response.data && response.data.message) ? response.data.message : 'Unknown error';
                    $urlsList.html('<p>Error loading issues: ' + errorMsg + '</p>');
                }
            },
            error: function() {
                $urlsList.html('<p>Error loading issues. Please try again.</p>');
            }
        });
    }
    
    /**
     * Display URLs in the container
     * @param {jQuery} $container The container element
     * @param {Array} urls The URLs to display
     * @param {number} page Current page number
     * @param {number} perPage Items per page
     * @param {boolean} showProblems Whether to show the Problem column
     */
    function displayUrls($container, urls, page, perPage, showProblems) {
        if (!urls || urls.length === 0) {
            $container.html('<p>No issues found.</p>');
            return;
        }
        
        var html = '<table class="rank-urls-table">';
        
        // Table header
        html += '<thead><tr><th>#</th>';
        html += '<th>URL</th>';
        if (showProblems) {
            html += '<th>Problem</th>';
        }
        html += '</tr></thead><tbody>';
        
        // Calculate the starting index based on the current page
        var startIndex = (page - 1) * perPage;
        
        urls.forEach(function(item, index) {
            html += '<tr>';
            html += '<td>' + (startIndex + index + 1) + '</td>';
            
            html += '<td><a href="' + escapeHtml(item.url) + '" target="_blank">' + escapeHtml(item.url) + '</a></td>';
            
            if (showProblems) {
                // Show the problem identifier - use basename for image URLs, full URL for broken links
                var problemDisplay = item.problem || '';
                if (problemDisplay) {
                    // Try to get basename for display, but show full URL on hover
                    var basename = problemDisplay.split('/').pop().split('?')[0];
                    if (basename && basename !== problemDisplay) {
                        problemDisplay = '<span title="' + escapeHtml(item.problem) + '">' + escapeHtml(basename) + '</span>';
                    } else {
                        problemDisplay = '<span title="' + escapeHtml(item.problem) + '">' + escapeHtml(item.problem) + '</span>';
                    }
                }
                html += '<td class="rank-problem-cell">' + problemDisplay + '</td>';
            }
            
            html += '</tr>';
        });
        
        html += '</tbody></table>';
        $container.html(html);
    }
    
    /**
     * Escape HTML special characters
     * @param {string} str The string to escape
     * @returns {string} Escaped string
     */
    function escapeHtml(str) {
        if (!str) return '';
        var div = document.createElement('div');
        div.appendChild(document.createTextNode(str));
        return div.innerHTML;
    }
    
    /**
     * Update pagination controls
     * @param {jQuery} $container The pagination container
     * @param {number} currentPage The current page
     * @param {number} totalPages The total number of pages
     * @param {Function} callback The callback function when a page is clicked
     */
    function updatePagination($container, currentPage, totalPages, callback) {
        if (totalPages <= 1) {
            $container.empty();
            return;
        }
        
        var html = '<div class="rank-pagination-controls">';
        
        // Previous button
        if (currentPage > 1) {
            html += '<button class="button" data-page="' + (currentPage - 1) + '">« Previous</button>';
        } else {
            html += '<button class="button" disabled>« Previous</button>';
        }
        
        // Page info
        html += '<span class="rank-pagination-info">Page ' + currentPage + ' of ' + totalPages + '</span>';
        
        // Next button
        if (currentPage < totalPages) {
            html += '<button class="button" data-page="' + (currentPage + 1) + '">Next »</button>';
        } else {
            html += '<button class="button" disabled>Next »</button>';
        }
        
        html += '</div>';
        $container.html(html);
        
        // Add click handlers
        $container.find('button[data-page]').on('click', function() {
            var page = parseInt($(this).data('page'));
            callback(page);
        });
    }
    
    /**
     * Add CSS styles for the modal
     */
    function addModalStyles() {
        var styles = 
            '.rank-modal {' +
                'display: none;' +
                'position: fixed;' +
                'z-index: 100000;' +
                'left: 0;' +
                'top: 0;' +
                'width: 100%;' +
                'height: 100%;' +
                'overflow: auto;' +
                'background-color: rgba(0, 0, 0, 0.4);' +
            '}' +
            
            '.rank-modal-content {' +
                'background-color: #fefefe;' +
                'margin: 5% auto;' +
                'padding: 20px;' +
                'border: 1px solid #888;' +
                'width: 80%;' +
                'max-width: 900px;' +
                'max-height: 80vh;' +
                'overflow-y: auto;' +
            '}' +
            
            '.rank-modal-close {' +
                'color: #aaa;' +
                'float: right;' +
                'font-size: 28px;' +
                'font-weight: bold;' +
                'cursor: pointer;' +
            '}' +
            
            '.rank-modal-close:hover,' +
            '.rank-modal-close:focus {' +
                'color: black;' +
                'text-decoration: none;' +
                'cursor: pointer;' +
            '}' +
            
            '.rank-urls-table {' +
                'width: 100%;' +
                'border-collapse: collapse;' +
                'margin-top: 10px;' +
            '}' +
            
            '.rank-urls-table th,' +
            '.rank-urls-table td {' +
                'padding: 8px;' +
                'text-align: left;' +
                'border-bottom: 1px solid #ddd;' +
            '}' +
            
            '.rank-urls-table th {' +
                'background-color: #f7f7f7;' +
                'font-weight: 600;' +
            '}' +
            
            '.rank-problem-cell {' +
                'max-width: 250px;' +
                'word-break: break-all;' +
                'font-family: monospace;' +
                'font-size: 12px;' +
            '}' +
            
            '.rank-pagination-controls {' +
                'display: flex;' +
                'justify-content: space-between;' +
                'align-items: center;' +
                'margin-top: 20px;' +
            '}' +
            
            '.rank-pagination-info {' +
                'text-align: center;' +
            '}' +
            
            '.rank-issue-line {' +
                'margin-bottom: 4px;' +
                'line-height: 2.4;' +
            '}' +
            
            '.rank-issue-line-total {' +
                'font-weight: 600;' +
                'font-size: 14px;' +
                'margin-bottom: 24px;' +
            '}' +
            
            '.rank-issue-line-fixed,' +
            '.rank-issue-line-remaining {' +
                'font-size: 13px;' +
                'padding-left: 4px;' +
            '}' +
            
            '.rank-already-fixed-infobox {' +
                'background-color: #fff8e5;' +
                'border-left: 4px solid #ffb900;' +
                'padding: 16px 20px;' +
                'margin-top: 16px;' +
                'font-size: 13px;' +
                'line-height: 1.5;' +
                'max-width: 760px;' +
            '}' +
            
            '.rank-infobox-title {' +
                'font-size: 14px;' +
                'font-weight: 700;' +
                'margin-bottom: 10px;' +
                'padding-bottom: 8px;' +
                'border-bottom: 1px solid #ffe0a0;' +
            '}' +
            
            'ul.rank-infobox-list {' +
                'margin: 0 !important;' +
                'padding-left: 18px !important;' +
                'list-style-type: disc !important;' +
            '}' +
            
            'ul.rank-infobox-list li {' +
                'margin-bottom: 6px;' +
                'padding-left: 4px;' +
                'list-style-type: disc !important;' +
                'display: list-item !important;' +
            '}' +
            
            'ul.rank-infobox-list li:last-child {' +
                'margin-bottom: 0;' +
            '}';
        
        $('<style>').text(styles).appendTo('head');
    }
    
    /**
     * Remove the "View URLs" button, fixed link, counts display, and infobox.
     * Called when processing starts (Manual or Automatic) to clean up the pre-processing UI.
     */
    function removeViewUrlsButton() {
        // Remove any existing buttons/links
        $('.rank-view-urls-link, .rank-view-fixed-link').remove();
        // Remove the infobox
        $('.rank-already-fixed-infobox').remove();
        // Clear the counts display (3-line total/fixed/remaining)
        if (rank.elements.counts && rank.elements.counts.length) {
            rank.elements.counts.empty();
        }
    }
    
    /**
     * Add the yellow infobox below the processing buttons
     * Explains why already-fixed issues still appear as problems in Morningscore
     * Should be called after buttons are shown so it appears below them
     */
    function addInfobox() {
        // Remove any existing infobox first
        $('.rank-already-fixed-infobox').remove();
        
        var fixedCount = (rank.state && rank.state.alreadyFixedCount) || 0;
        
        // Show whenever there are already-fixed issues (even when all are fixed, i.e. 0 remaining)
        if (fixedCount <= 0) {
            return;
        }
        
        // Format crawled_at date
        var crawledAtStr = 'unknown';
        if (rank.state && rank.state.crawledAt) {
            try {
                var crawledDate = new Date(rank.state.crawledAt);
                crawledAtStr = crawledDate.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' }) +
                               ' at ' + crawledDate.toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit' });
            } catch (e) {
                crawledAtStr = rank.state.crawledAt;
            }
        }
        
        var html = '<div class="rank-already-fixed-infobox">';
        html += '<div class="rank-infobox-title">Why ' + fixedCount + ' fixes still appear as problems:</div>';
        html += '<ul class="rank-infobox-list">';
        html += '<li>Last full Health scan was done ' + crawledAtStr + ' - is that after you fixed these ' + fixedCount + ' problems?</li>';
        html += '<li>Your site uses cache and your changes are not live yet</li>';
        html += '<li>Issues with your WordPress site that prevents these ' + fixedCount + ' fixes from working</li>';
        html += '<li>Incompatibility. Expect fixes to work in &asymp;90% of cases. Manually fix or <a href="https://help.morningscore.io/en/article/how-to-make-the-tool-ignore-health-issues-oyh8cj/" target="_blank">put in bin 🗑️</a></li>';
        html += '</ul>';
        html += '</div>';
        
        var $infobox = $(html);
        
        // Insert after the buttons/credits warning, or after counts if buttons are hidden
        var $creditsWarning = $('.rank-credits-warning');
        if ($creditsWarning.length) {
            $creditsWarning.after($infobox);
        } else if (rank.elements.autoProcessButton && rank.elements.autoProcessButton.length && rank.elements.autoProcessButton.is(':visible')) {
            rank.elements.autoProcessButton.after($infobox);
        } else if (rank.elements.startButton && rank.elements.startButton.length && rank.elements.startButton.is(':visible')) {
            rank.elements.startButton.after($infobox);
        } else if (rank.elements.counts && rank.elements.counts.length) {
            // Buttons are hidden (e.g. all issues already fixed) - place after counts
            rank.elements.counts.after($infobox);
        }
    }
    
    /**
     * Remove the infobox
     */
    function removeInfobox() {
        $('.rank-already-fixed-infobox').remove();
    }
    
    // Expose public methods
    rank.urlListModal = {
        init: init,
        addViewUrlsButton: addViewUrlsButton,
        addIssueCountsDisplay: addIssueCountsDisplay,
        removeViewUrlsButton: removeViewUrlsButton,
        showUrlsModal: showUrlsModal,
        showFixedIssuesModal: showFixedIssuesModal,
        addInfobox: addInfobox,
        removeInfobox: removeInfobox
    };
    
})(jQuery, window.rankLocalizedData);
