/**
 * API Key Notice Module
 * Ensures the API key notice is displayed when needed
 */
(function($, rank) {
    'use strict';
    
    // Function to add the API key notice
    function addApiKeyNotice() {
        // Check if we're on the correct admin page
        const urlParams = new URLSearchParams(window.location.search);
        const currentPage = urlParams.get('page');
        
        if (currentPage !== 'rank-admin') {
            console.log('RANK AI: Not on rank-admin page, skipping notice check');
            return; // Only run on our plugin's admin page
        }
        
        // Check if we're on the bulk_fix tab
        const currentTab = urlParams.get('tab') || 'bulk_fix'; // Default to bulk_fix if no tab specified
        
        if (currentTab !== 'bulk_fix') {
            console.log('RANK AI: Not on bulk_fix tab, skipping notice');
            return; // Only show notice on the bulk_fix tab
        }
        
        // Check if the API key is set (using the data attribute on the .wrap div)
        // Be more specific - look for the wrap div that has the data-api-key-set attribute
        const $wrapWithApiKey = $('.wrap[data-api-key-set]');
        
        if ($wrapWithApiKey.length === 0) {
            console.log('RANK AI: No .wrap element with data-api-key-set attribute found');
            // Fallback: check if there's any content in the bulk_fix tab that indicates API key is set
            if ($('#rank-health-issues-container').length > 0 || $('.rank-section').length > 0) {
                console.log('RANK AI: Found bulk_fix content, assuming API key is set');
                return;
            }
            console.log('RANK AI: No bulk_fix content found either, will show notice');
        } else {
            const apiKeySetValue = $wrapWithApiKey.data('api-key-set');
            const isApiKeySet = apiKeySetValue === 'yes';
            
            console.log('RANK AI: API key status check - value:', apiKeySetValue, 'isSet:', isApiKeySet);
            
            if (isApiKeySet) {
                console.log('RANK AI: API key is set, no notice needed');
                return;
            }
        }
        
        // If we reach here, we need to show the notice
        // Check if notice already exists to avoid duplicates
        if ($('.rank-api-key-notice').length > 0) {
            console.log('RANK AI: API key notice already exists, skipping');
            return;
        }
        
        // Get the current URL and replace or add the tab parameter
        const currentUrl = new URL(window.location.href);
        currentUrl.searchParams.set('tab', 'api_config');
        
        // Create the notice with a link to the API config tab
        const $notice = $('<div class="notice notice-warning rank-api-key-notice" style="display: block !important; margin-top: 10px;">' +
            '<p><a href="' + currentUrl.toString() + '">Add API key in settings</a> to start fixing issues 🚀</p>' +
            '</div>');
        
        // Insert it after the tab navigation
        $('.nav-tab-wrapper').after($notice);
        console.log('RANK AI: API key notice added');
    }
    
    // Run when the document is ready
    $(document).ready(function() {
        // Add a small delay to ensure other scripts have run and DOM is fully loaded
        setTimeout(function() {
            try {
                addApiKeyNotice();
            } catch (error) {
                console.error('RANK AI: Error in addApiKeyNotice:', error);
            }
        }, 100);
    });
    
})(jQuery, window.rank = window.rank || {});